const CACHE_NAME = 'radio-cache-v2';
const STATIC_ASSETS = [
  '/css/styles.css',
  '/js/script.js',
  '/img/logo.png',
  '/img/72.png',
  '/img/96.png',
  '/img/192.png',
  '/img/512.png',
  '/default-cover.jpg',
  '/index.html',      // Para fallback navegacional offline
  '/offline.html'     // Página offline (debes crearla)
];

// Instalación: cachear recursos estáticos
self.addEventListener('install', event => {
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then(cache => cache.addAll(STATIC_ASSETS))
      .then(() => self.skipWaiting())
  );
});

// Activación: limpiar cachés antiguos
self.addEventListener('activate', event => {
  event.waitUntil(
    caches.keys().then(keys =>
      Promise.all(
        keys.filter(key => key !== CACHE_NAME)
          .map(key => caches.delete(key))
      )
    ).then(() => self.clients.claim())
  );
});

// Interceptar peticiones
self.addEventListener('fetch', event => {
  const request = event.request;

  // Para navegación, usar estrategia network-first con fallback offline.html
  if (request.mode === 'navigate') {
    event.respondWith(
      fetch(request)
        .then(response => {
          // Opcional: actualizar cache de index.html
          const copy = response.clone();
          caches.open(CACHE_NAME).then(cache => cache.put('/index.html', copy));
          return response;
        })
        .catch(() => caches.match('/offline.html'))
    );
    return;
  }

  // Para streaming de audio: siempre fetch en red
  if (request.destination === 'audio') {
    event.respondWith(fetch(request));
    return;
  }

  // Para otros recursos estáticos: cache-first con actualización en segundo plano
  event.respondWith(
    caches.match(request).then(cachedResponse => {
      if (cachedResponse) {
        // Actualizar caché en background (sin bloquear respuesta)
        fetch(request).then(networkResponse => {
          if (
            networkResponse &&
            networkResponse.status === 200 &&
            request.method === 'GET' &&
            STATIC_ASSETS.includes(new URL(request.url).pathname)
          ) {
            caches.open(CACHE_NAME).then(cache => cache.put(request, networkResponse.clone()));
          }
        }).catch(() => {
          // Aquí se puede loggear o ignorar errores de fetch background
        });
        return cachedResponse;
      }

      // Si no está en caché, buscar en red y cachear si es recurso estático
      return fetch(request).then(networkResponse => {
        if (
          networkResponse &&
          networkResponse.status === 200 &&
          request.method === 'GET' &&
          STATIC_ASSETS.includes(new URL(request.url).pathname)
        ) {
          caches.open(CACHE_NAME).then(cache => cache.put(request, networkResponse.clone()));
        }
        return networkResponse;
      }).catch(() => {
        // Fallback opcional: devolver imagen offline o algo similar
        if (request.destination === 'image') {
          return caches.match('/img/default-cover.png'); // Imagen genérica offline
        }
        return new Response('Conexión no disponible.', { status: 503, statusText: 'Offline' });
      });
    })
  );
});
