const CACHE_NAME = 'radio-cache-v2';
const STATIC_ASSETS = [
  '/css/styles.css',
  '/js/script.js',
  '/img/logo.png',
  '/img/72.png',
  '/img/96.png',
  '/img/192.png',
  '/img/512.png',
  '/default-cover.jpg',
  '/index.html',      // For offline navigation fallback
  '/offline.html'     // Offline page (you need to create it)
];

// Install: cache static assets
self.addEventListener('install', event => {
  event.waitUntil(
    caches.open(CACHE_NAME)
      .then(cache => cache.addAll(STATIC_ASSETS))
      .then(() => self.skipWaiting())
  );
});

// Activate: clean up old caches
self.addEventListener('activate', event => {
  event.waitUntil(
    caches.keys().then(keys =>
      Promise.all(
        keys.filter(key => key !== CACHE_NAME)
          .map(key => caches.delete(key))
      )
    ).then(() => self.clients.claim())
  );
});

// Intercept requests
self.addEventListener('fetch', event => {
  const request = event.request;

  // For navigation, use network-first strategy with offline.html fallback
  if (request.mode === 'navigate') {
    event.respondWith(
      fetch(request)
        .then(response => {
          // Optional: update index.html cache
          const copy = response.clone();
          caches.open(CACHE_NAME).then(cache => cache.put('/index.html', copy));
          return response;
        })
        .catch(() => caches.match('/offline.html'))
    );
    return;
  }

  // For audio streaming: always fetch from network
  if (request.destination === 'audio') {
    event.respondWith(fetch(request));
    return;
  }

  // For other static resources: cache-first with background update
  event.respondWith(
    caches.match(request).then(cachedResponse => {
      if (cachedResponse) {
        // Update cache in the background (without blocking response)
        fetch(request).then(networkResponse => {
          if (
            networkResponse &&
            networkResponse.status === 200 &&
            request.method === 'GET' &&
            STATIC_ASSETS.includes(new URL(request.url).pathname)
          ) {
            caches.open(CACHE_NAME).then(cache => cache.put(request, networkResponse.clone()));
          }
        }).catch(() => {
          // Optional: log or ignore background fetch errors
        });
        return cachedResponse;
      }

      // If not in cache, fetch from network and cache if static resource
      return fetch(request).then(networkResponse => {
        if (
          networkResponse &&
          networkResponse.status === 200 &&
          request.method === 'GET' &&
          STATIC_ASSETS.includes(new URL(request.url).pathname)
        ) {
          caches.open(CACHE_NAME).then(cache => cache.put(request, networkResponse.clone()));
        }
        return networkResponse;
      }).catch(() => {
        // Optional fallback: return offline image or similar
        if (request.destination === 'image') {
          return caches.match('/img/default-cover.png'); //
